<?php

namespace App\Http\Controllers\Doctor;

use App\Http\Controllers\Controller;
use App\Models\DeudaPaciente;
use App\Models\DiarioPaciente;
use App\Models\Paciente;
use App\Models\PagoDoctor;
use App\Models\Procedimiento;
use App\Models\RecordatorioPaciente;
use App\Models\Tratamiento;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PacienteDoctorController extends Controller
{



    function showPaciente(Paciente $paciente, Request $request)
    {

        $procedimientos = Procedimiento::where('paciente_id', $paciente->id)
            ->where('hoja', $request->hoja)
            ->where('fecha', null)
            ->orderBy('id', 'asc')
            ->get();

        $users = User::all();

        // hacer que vengan los procediminetos realizados en este paciente

        $procedimientos_doc = Procedimiento::where('paciente_id', $paciente->id)
            ->where('hoja', $request->hoja)
            ->where('fecha', '!=', null)
            ->where('doctor_id', session()->get('doctor')['id'])
            ->orderBy('id', 'asc')
            ->get();

        $recordatorios = RecordatorioPaciente::where('paciente_id', $paciente->id)->get();

        return view('doctor.pacientes.show', compact('paciente', 'procedimientos', 'users', 'procedimientos_doc', 'recordatorios'));
    }



    function setPagosHistorialAndMore(Request $request)
    {
        // dd($request->all());
        // 1. Validamos los datos (Ahora procedimientos_ids es un array)
        $request->validate([
            'procedimientos_ids' => 'nullable|array',     // El array de checkboxes
            'procedimientos_ids.*' => 'exists:procedimientos,id', // Cada ID debe existir
            'doctor_id' => 'required|exists:doctores,id',
            'paciente_id' => 'required|exists:pacientes,id',
            'asistente' => 'nullable|string|max:255',
            'recomendacion' => 'nullable|string',         // Paso 2 del Wizard
            'descripcion' => 'required|string',           // La nota del Diario
            'proxima' => 'nullable|string',               // Próxima cita
        ]);

        // Usamos una Transacción para asegurar que todo se guarde o nada se guarde (Seguridad)
        return DB::transaction(function () use ($request) {

            $total_deuda_sesion = 0;
            $piezas_tratadas = [];
            $nombre_doctor = session()->get('doctor')['nombre'];

            // --- PARTE A: PROCESAR TRATAMIENTOS (Si seleccionó alguno) ---
            if ($request->has('procedimientos_ids')) {

                foreach ($request->procedimientos_ids as $proc_id) {
                    $presu = Procedimiento::findOrFail($proc_id); // Buscamos el tratamiento individual

                    // 1. Actualizar el procedimiento
                    $presu->doctor_id = $request->doctor_id;
                    $presu->estado_pago = true;
                    $presu->fecha = date('Y-m-d');
                    $presu->asistio = $request->asistente;
                    $presu->recomendacion = $request->recomendacion; // Recomendación global
                    $presu->bloqueo = 'NO';
                    $presu->save();

                    // Guardamos la pieza para ponerla en el diario después
                    $piezas_tratadas[] = $presu->pieza;

                    // 2. Registrar Pago al Doctor (Historial)
                    $pago = new PagoDoctor();
                    $pago->doctor_id = $request->doctor_id;
                    $pago->paciente_id = $request->paciente_id;
                    $pago->tratamiento_id = $presu->tratamiento_id;
                    $pago->pieza = $presu->pieza;
                    $pago->save();

                    // 3. Registrar Deuda individual
                    $deuda = new DeudaPaciente();
                    $deuda->paciente_id = $request->paciente_id;
                    $deuda->doctor_id = $request->doctor_id;
                    $deuda->tratamiento_id = $presu->tratamiento_id;
                    $deuda->pieza = $presu->pieza;
                    $deuda->valor = $presu->valor;
                    $deuda->save();

                    // Sumamos al acumulador de esta sesión
                    $total_deuda_sesion += $presu->valor;
                }
            }

            // --- PARTE B: ACTUALIZAR DEUDA TOTAL DEL PACIENTE ---
            if ($total_deuda_sesion > 0) {
                $paciente = Paciente::find($request->paciente_id);
                $paciente->deuda = $paciente->deuda + $total_deuda_sesion;
                $paciente->save();
            }

            // --- PARTE C: GUARDAR DIARIO (Paso 2 del Wizard) ---
            // Esto ahora se guarda SIEMPRE, aunque no haya tratamientos (solo nota de evolución)
            $diario = new DiarioPaciente();
            $diario->paciente_id = $request->paciente_id;
            $diario->doctor = $nombre_doctor;
            $diario->descripcion = $request->descripcion; // Lo que escribiste en el textarea
            $diario->proxima = $request->proxima;         // La próxima cita

            // Convertimos el array de piezas a string: "18, 21, 24" o "Consulta"
            $diario->piezas = count($piezas_tratadas) > 0 ? implode(', ', $piezas_tratadas) : 'S/P';
            $diario->save();

            // Recordatorio de paciente
            $reco = new RecordatorioPaciente();
            $reco->paciente_id = $request->paciente_id;
            $reco->descripcion = $request->mensaje_recordatorio;
            $reco->fecha = date('Y-m-d');
            $reco->save();

            // --- PARTE D: LOGS ---
            storeControl('Registro de atención múltiple y diario', $nombre_doctor);

            return back()->with(['info' => 'Cambios y tratamientos registrados con éxito', 'color' => 'success']);
        });
    }
}
