<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeudaPaciente;
use App\Models\Procedimiento;
use App\Models\SalidaInsumo;
use App\Models\Tratamiento;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class TratamientoController extends Controller
{

    public function index()
    {
        abort_if(Gate::denies('tratamiento_listado'), Response::HTTP_FORBIDDEN, 'Forbidden');
        $tratamientos  = Tratamiento::all();
        return view('admin.tratamiento.index', ['tratamientos' => $tratamientos]);
    }

    public function create()
    {
        abort_if(Gate::denies('tratamiento_store'), Response::HTTP_FORBIDDEN, 'Forbidden');
        return view('admin.tratamiento.create');
    }

    public function store(Request $request)
    {
        abort_if(Gate::denies('tratamiento_store'), Response::HTTP_FORBIDDEN, 'Forbidden');
        Tratamiento::create($request->all());
        return back()->with(['info' => 'Tratamiento Creado Con Exito', 'color' => 'success']);
    }

    public function show($id)
    {
        abort_if(Gate::denies('tratamiento_update'), Response::HTTP_FORBIDDEN, 'Forbidden');

        $trata = Tratamiento::find($id);

        // dd($trata);
        return view('admin.tratamiento.show', ['tratamiento' => $trata]);
    }

    /**
     * Actualiza un tratamiento
     *
     * @param  Request $request
     * @param  Tratamiento $tratamiento
     * @return \Illuminate\Http\Response
     */

    public function update(Request $request, Tratamiento $tratamiento)
    {
        abort_if(Gate::denies('tratamiento_update'), Response::HTTP_FORBIDDEN, 'Forbidden');

        $tratamiento->update($request->all());
        return back()->with(['info' => 'Tratamiento Actualizado con Exito', 'color' => 'warning']);
    }

    public function destroy($id)
    {
        abort_if(Gate::denies('tratamiento_update'), Response::HTTP_FORBIDDEN, 'Forbidden');
        Tratamiento::find($id)->delete();
        return back()->with(['info' => 'Tratamiento  Eliminado con Exito', 'color' => 'danger']);
    }

    public function reportes()
    {
        $tratamiento = Tratamiento::all();
        return view('admin.tratamiento.reportes', compact('tratamiento'));
    }

    public function printAll()
    {
        return view('reports.tratamientos.all', ['tratamientos' => Tratamiento::all()]);
    }

    // retorna los tratamientos de mas a menos
    function masToMinus(Request $request)
    {
        $data = DeudaPaciente::join('tratamientos as tr', 'deuda_pacientes.tratamiento_id', '=', 'tr.id')
            ->select(DB::raw('count(tratamiento_id) as trid'), 'tr.nombre', 'tr.descripcion', 'tr.tipo')
            ->groupby('deuda_pacientes.tratamiento_id')
            ->whereBetween('deuda_pacientes.created_at', [$request->inicio, $request->fin])
            ->orderBy('trid', 'desc')
            ->get();
        return view('reports.tratamientos.minusandmore', compact('data', 'request'));
    }

    function faltantesOperacion(Request $request)
    {

        $data = Procedimiento::where('estado_pago', '0')->where('tratamiento_id', $request->tratamiento_id)->whereBetween('created_at', [$request->inicio, $request->fin])->groupby('tratamiento_id')->get();

        $inicio = $request->inicio;
        $fin = $request->fin;


        return view('reports.tratamientos.faltantes', compact('data', 'inicio', 'fin'));
    }


    // TODO: VER SOLO LE EFECTUADO, EN TRATAMIENTOS


    function totalTratamiento(Request $request)
    {

        // TODO: ver solo abonos

        $data = null;
        if ($request->tratamiento_id == 'TODO') {
            $data = DeudaPaciente::whereBetween('deuda_pacientes.created_at', [$request->inicio, $request->fin])->orderby('tratamiento_id', 'asc')->get();
        } else {
            $data = DeudaPaciente::whereBetween('deuda_pacientes.created_at', [$request->inicio, $request->fin])->where('tratamiento_id', $request->tratamiento_id)->get();
        }

        return view('reports.tratamientos.totaltesTratamiento', compact('data'));
    }




    // reporte de salidas de insumos en tratamientos viejos
    // function salidadeinsumoentratamiento(Request $request)
    // {
    // $data = Procedimiento::whereBetween('fecha', [$request->inicio, $request->fin])->orderby('asistio', 'desc')->get();
    // return view('reports.tratamientos.salidadeinsumoentratamiento', compact('data'));
    // }



    // retona el reporte de en que tratamiento si se realizaron salida de insumos

    /**
     * Muestra el reporte de salida de insumos en tratamientos.
     * Optimizado para evitar problemas de rendimiento N+1.
     *
     * @param Request $request - Contiene las fechas 'inicio' y 'fin' del filtro.
     * @return \Illuminate\View\View
     */
    public function salidadeinsumoentratamiento(Request $request)
    {
        // Iniciamos la construcción de la consulta sobre el modelo Procedimiento.
        // Usar 'query()' es una buena práctica para iniciar el Query Builder.
        $data = Procedimiento::query()

            // 1. IMPORTANTE: Seleccionamos explícitamente todas las columnas de la tabla 'procedimientos'.
            // Si no hacemos esto, al usar 'addSelect' más abajo, podríamos perder los datos originales del modelo.
            ->select('procedimientos.*')

            // 2. SOLUCIÓN N+1 (Relaciones): Eager Loading (Carga Ansiosa).
            // Pre-carga los datos de 'paciente' y 'tratamiento' en 2 consultas adicionales
            // en lugar de 1 consulta por cada fila del reporte. Esto acelera la carga masivamente.
            ->with(['paciente', 'tratamiento'])

            // 3. SOLUCIÓN N+1 (Lógica de Negocio): Subconsulta optimizada.
            // Agregamos una columna "virtual" llamada 'tiene_salida' a cada resultado.
            // Esto elimina la necesidad de llamar al Helper 'searSalidaInsumo' dentro del bucle en la vista.
            ->addSelect([
                'tiene_salida' => SalidaInsumo::selectRaw('1') // Solo seleccionamos '1' por eficiencia (no necesitamos traer toda la fila).

                    // Relacionamos la subconsulta con la consulta principal:
                    // "Busca salidas de insumo donde el paciente_id coincida con el paciente_id del procedimiento actual"
                    ->whereColumn('salida_insumos.paciente_id', 'procedimientos.paciente_id')

                    // Comparamos las fechas.
                    // 'whereRaw' nos permite usar funciones de SQL.
                    // DATE(...) convierte el Timestamp (2023-10-25 14:30:00) a Fecha pura (2023-10-25)
                    // para poder compararla con 'procedimientos.fecha'.
                    ->whereRaw('DATE(salida_insumos.created_at) = procedimientos.fecha')

                    // Optimización: En cuanto encuentre 1 registro, deja de buscar.
                    // Solo nos interesa saber SI existe, no cuántos hay.
                    ->limit(1)
            ])

            // 4. Filtros del usuario:
            // Filtra los procedimientos basándose en el rango de fechas recibido del formulario.
            ->whereBetween('fecha', [$request->inicio, $request->fin])

            // 5. Ordenamiento:
            // Ordena los resultados por la columna 'asistio' de forma descendente.
            ->orderBy('asistio', 'desc')

            // 6. Ejecución:
            // Finaliza la construcción del query, ejecuta el SQL en la base de datos y obtiene la Colección de resultados.
            ->get();

        // Retornamos la vista pasando la variable $data compactada.
        return view('reports.tratamientos.salidadeinsumoentratamiento', compact('data'));
    }

    // cambia el estado de bloqueado a desbloqueado de un procedimiento
    public function desbloquear($procedimiento_id)
    {
        $procedimiento = Procedimiento::find($procedimiento_id);
        $procedimiento->update(['bloqueo' => NULL]);
        return back();
    }
}
