<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Insumo;
use App\Models\LoteInsumo;
use App\Models\SalidaInsumo;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class InsumoController extends Controller
{

    public function index()
    {
        abort_if(Gate::denies('insumo_list'), Response::HTTP_FORBIDDEN, 'Forbidden');

        return view('admin.insumos.index', ['insumos' => Insumo::all()]);
    }

    public function create()
    {
        abort_if(Gate::denies('insumo_store'), Response::HTTP_FORBIDDEN, 'Forbidden');

        return view('admin.insumos.create');
    }

    public function show($id)
    {
        abort_if(Gate::denies('insumo_update'), Response::HTTP_FORBIDDEN, 'Forbidden');
        return view('admin.insumos.show', ['insumo' => Insumo::find($id)]);
    }

    public function store(Request $request)
    {
        // 1. Verificación de Permisos
        abort_if(Gate::denies('insumo_store'), Response::HTTP_FORBIDDEN, 'Forbidden');

        // 2. Creación del Insumo principal
        $insumo = Insumo::create($request->except('lotes'));

        // 3. Creación de los Lotes asociados
        foreach ($request->lotes as $lote) {
            // Asegurarse de que el lote no esté vacío antes de intentar crear (manejo básico de lotes vacíos)
            if (!empty($lote['cantidad'])) {
                LoteInsumo::create([
                    'insumo_id'         => $insumo->id,
                    'cantidad'          => $lote['cantidad'],
                    'fecha_vencimiento' => $lote['fecha_vencimiento'] ?? null,
                    'factura'           => $lote['factura'] ?? null,
                ]);
            }
        }

        // 4. Registro de control
        storeControl('registro un nuevo insumo');

        // 5. Redirección PRG (POST-REDIRECT-GET)
        // Redirigimos a una ruta que usa el método GET (e.g., el índice de insumos),
        // para evitar el doble envío del formulario si el usuario recarga la página.

        // **Asegúrate de que tienes una ruta 'insumos.index' definida.**
        return redirect()->route('insumos.index')
            ->with(['info' => 'Insumo registrado con éxito', 'color' => 'success']);
    }

    public function update(Request $request, Insumo $insumo)
    {
        abort_if(Gate::denies('insumo_update'), Response::HTTP_FORBIDDEN, 'Forbidden');

        // 1. Actualizar el Insumo principal
        $insumo->update($request->except('lotes'));

        // Colección de IDs de lotes que vienen en la petición (excluyendo null/cero)
        $lotesRequestIds = collect($request->lotes)
            ->pluck('id')
            ->filter();

        // Colección de IDs de lotes que YA existen en la base de datos para este insumo
        $lotesExistentesIds = $insumo->lotes()->pluck('id');


        // 2. Crear o Actualizar lotes
        // 2. Crear o Actualizar lotes (Optimizado)
        foreach ($request->lotes as $loteData) {
            // Esto es ahora seguro porque el ID está presente para lotes existentes,
            // y ausente (o null) para lotes nuevos.
            $insumo->lotes()->updateOrCreate(
                // Criterio de búsqueda: Busca por 'id' si existe, o por 'insumo_id' si es nuevo.
                ['id' => $loteData['id'] ?? null],
                // Valores a crear o actualizar:
                [
                    'cantidad'          => $loteData['cantidad'],
                    'fecha_vencimiento' => $loteData['fecha_vencimiento'],
                    'factura'           => $loteData['factura'],
                ]
            );
        }

        // 3. Eliminar lotes que YA NO están en la petición y NO HAN SIDO USADOS
        $lotesAEliminarIds = $lotesExistentesIds->diff($lotesRequestIds);

        if ($lotesAEliminarIds->isNotEmpty()) {
            // Encontrar los lotes que deben eliminarse Y que no tienen salidas asociadas
            LoteInsumo::whereIn('id', $lotesAEliminarIds)
                // Esto usa la relación 'salidas()' definida en el modelo
                ->doesntHave('salidas')
                ->delete();
        }

        return back()->with(['info' => 'Insumo actualizado con éxito', 'color' => 'warning']);
    }

    public function destroy(Insumo $insumo)
    {
        try {
            abort_if(Gate::denies('insumo_delte'), Response::HTTP_FORBIDDEN, 'Forbidden');

            $insumo->lotes()->delete();

            $insumo->delete();
            storeControl('elimino un insumo');

            return back()->with(['info' => 'insumo eliminado con exito', 'color' => 'danger']);
        } catch (\Throwable $th) {
            return back()->with(['info' => 'No se puede eliminar el insumo porque tiene salidas asociadas.', 'color' => 'danger']);
        }
    }

    public function reportPrint()
    {
        return view('reports.insumos.all', ['insumos' => Insumo::all()]);
    }


    // retona la vista para visulizar y seleccionar reportes
    public function reportPrints()
    {
        return view('admin.insumos.reporte');
    }


    // ------- salida de insumos

    // retorna la vista de salidas de Insumos
    public function salidas()
    {

        $salidas = DB::table('salida_insumos as si')
            ->join('insumos as in', 'si.insumo_id', '=', 'in.id')
            ->join('users as us', 'si.user_id', '=', 'us.id')
            ->select(
                'si.cantidad',
                'si.clinica',
                'si.created_at',
                'si.id',
                'in.nombre as insumo',
                'in.id as insumo_id',
                'us.name'
            )
            ->orderBy('si.id', 'desc')
            ->paginate(30);

        alert('Salidas de Insumos sin Pacientes');

        return view('admin.insumos.salidas.index', ['salidas' => $salidas, 'insumos' => Insumo::all()]);
    }

    // guarda la salida de un insumo
    public function salidaStore(Request $request)
    {
        $insumo = Insumo::find($request->insumo_id);

        if ($request->cantidad > $insumo->cantidad) {
            return back()->with(['info' => 'la cantidad supera el stock', 'color' => 'warning']);
        }
        SalidaInsumo::create($request->all());
        $insumo->cantidad -= $request->cantidad;
        $insumo->save();
        storeControl('registro una salida de insumos');

        return back()->with(['info' => 'salida de insumo registrado con exito', 'color' => 'success']);
    }


    // elimina un salida
    public function deleteSalida(Request $request)
    {
        SalidaInsumo::find($request->id)->delete();
        $insumo = Insumo::find($request->insumo_id);
        $insumo->cantidad += $request->cantidad;
        $insumo->save();
        storeControl('elimino un registro de insumo');

        return back()->with(['info' => 'salida de insumo eliminada con exito', 'color' => 'danger']);
    }

    // genera el reporte de salida de Insumos
    public function salidasReporteDate(Request $request)
    {
        $salidas = DB::table('salida_insumos as si')
            ->join('insumos as in', 'si.insumo_id', '=', 'in.id')
            ->join('users as us', 'si.user_id', '=', 'us.id')
            ->select(
                'si.cantidad',
                'si.clinica',
                'si.created_at',
                'si.id',
                'in.nombre as insumo',
                'in.id as insumo_id',
                'us.name'
            )
            ->orderBy('si.id', 'desc')
            ->whereBetween('si.created_at', [$request->desde, $request->hasta])
            ->get();
        return view('reports.insumos.salidas', ['salidas' => $salidas]);
    }


    // para elimiar el lote de insumo
    public function loteDestroy(LoteInsumo $lote)
    {


        // 2. Comprobar si el lote tiene salidas asociadas
        if ($lote->salidas()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'No se puede eliminar el lote porque tiene salidas registradas.',
            ], 403); // HTTP 403 Forbidden
        }

        // 3. Eliminar el Lote
        $lote->delete();

        // 4. Registrar control (opcional)
        // storeControl('elimino un lote de insumo');

        // 5. Devolver respuesta JSON de éxito
        return response()->json([
            'success' => true,
            'message' => 'Lote eliminado con éxito.',
        ]);
    }
}
