<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Abono;
use App\Models\Articulo;
use App\Models\Doctor;
use App\Models\Insumo;
use App\Models\LoteInsumo;
use App\Models\LoteStock;
use App\Models\MobiliarioEquipo;
use App\Models\Paciente;
use App\Models\PagoFactura;
use App\Models\Procedimiento;
use App\Models\RecordatorioGeneral;
use App\Models\Tratamiento;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        $paciente = $this->totalPaciente();
        $doctore = $this->totalDoctores();
        $agendados =  $this->agendadosParaHoy();
        $usuarios = $this->usuariosRegistrados();
        $medicamentos = $this->medicamentosRegistrados();
        $salidas = $this->totalMovimiento('s');
        $entradas = $this->totalMovimiento('e');
        $pagos = $this->pagosFac();
        $vencidos = $this->articulosVencidos();
        $minimos = $this->articulosminimo();
        $insumoVenciso = $this->insumosexpiraso();
        $insumoMinimo = $this->insumosminimo();
        $vacaciones = $this->vacaciones();
        $totalCaja = $this->totalCaja();
        $mantenimiento = $this->mantenimientoMobiliario();
        $cumpleaneros = $this->cumpleaneros();
        $recordatoriosGeneral = $this->recordatorioGenerales();
        $recPaciente = $this->recordatorioPaciente();

        // dd($insumoVenciso);


        $fechaHace6Meses = Carbon::now()->subMonths(6)->toDateString();

        $pacientes = Paciente::whereDate('updated_at', $fechaHace6Meses)->get();

        $tratamiento_ortodontia = Tratamiento::query()

            ->where('tipo2', 'ORTODONCIA')
            ->where('tipo', 'REAGENDA')
            // Aquí está la magia: Filtramos la relación, no el padre.
            ->with(['procedimientos' => function ($query) {
                // Solo cargamos los procedimientos que NO tienen fecha (pendientes)
                $query->whereNull('fecha');

                // Y cargamos al paciente de una vez para evitar N+1
                $query->with('paciente');
            }])

            // OPCIONAL: Si solo quieres ver tratamientos que TENGAN pendientes
            // y ocultar los que están al día, descomenta la siguiente línea:
            // ->whereHas('procedimientos', function ($q) {
            //     $q->whereNull('fecha');
            // })

            ->get();

        // dd($tratamiento_ortodontia);



        $tratamiento_endodoncia = Tratamiento::query()
            ->where('tipo2', 'ENDODONCIA')
            ->where('tipo', 'REAGENDA')

            // Aquí está la magia: Filtramos la relación, no el padre.
            ->with(['procedimientos' => function ($query) {
                // Solo cargamos los procedimientos que NO tienen fecha (pendientes)
                $query->whereNull('fecha');

                // Y cargamos al paciente de una vez para evitar N+1
                $query->with('paciente');
            }])

            // OPCIONAL: Si solo quieres ver tratamientos que TENGAN pendientes
            // y ocultar los que están al día, descomenta la siguiente línea:
            ->whereHas('procedimientos', function ($q) {
                $q->whereNull('fecha');
            })

            ->get();



        $tratamiento_maxilo = Tratamiento::query()
            ->where('tipo2', 'MAXILOFACIAL')
            ->where('tipo', 'REAGENDA')

            // Aquí está la magia: Filtramos la relación, no el padre.
            ->with(['procedimientos' => function ($query) {
                // Solo cargamos los procedimientos que NO tienen fecha (pendientes)
                $query->whereNull('fecha');

                // Y cargamos al paciente de una vez para evitar N+1
                $query->with('paciente');
            }])

            // OPCIONAL: Si solo quieres ver tratamientos que TENGAN pendientes
            // y ocultar los que están al día, descomenta la siguiente línea:
            ->whereHas('procedimientos', function ($q) {
                $q->whereNull('fecha');
            })

            ->get();


        return view('admin.home', ['paciente' => $paciente, 'doctore' => $doctore, 'agendados' => $agendados, 'usuarios' => $usuarios - 1, 'medicamentos' => $medicamentos, 'salidas' => $salidas, 'entradas' => $entradas, 'pagos' => $pagos, 'vencidos' => $vencidos, 'minimos' => $minimos, 'ivencidos' => $insumoVenciso, 'iminimos' => $insumoMinimo, 'vacaciones' => $vacaciones, 'totalCaja' => $totalCaja, 'mantenimientos' => $mantenimiento, 'cumpleaneros' => $cumpleaneros, 'generales' => $recordatoriosGeneral, 'recpaciente' => $recPaciente, 'pacientes' => $pacientes, 'tratamiento_ortodontia' => $tratamiento_ortodontia, 'tratamiento_endodoncia' => $tratamiento_endodoncia, 'tratamiento_maxilo' => $tratamiento_maxilo]);
    }


    public function totalPaciente()
    {
        return Paciente::count();
    }

    public function totalDoctores()
    {
        return Doctor::count();
    }

    public function agendadosParaHoy()
    {
        return DB::table('agendas as ag')
            ->join('doctores as doc', 'ag.doctor_id', '=', 'doc.id')
            ->select('doc.nombre', 'ag.paciente', 'ag.fecha', 'ag.hora', 'ag.tratamiento')
            ->whereDate('fecha', date('Y-m-d'))
            ->orderBy('fecha', 'asc')
            ->orderBy('hora', 'asc')
            ->get();
    }

    public function usuariosRegistrados()
    {
        return User::count();
    }

    public function medicamentosRegistrados()
    {
        return Articulo::count();
    }


    // retorna el total de un movimient de caja
    public function totalMovimiento($movimiento)
    {
        $movi =  DB::table('cajas')
            ->where('operacion', $movimiento)
            ->whereDate('fecha', date('Y-m-d'))
            ->get();
        $total = 0;

        foreach ($movi as $mv) {
            $total = $total + $mv->valor;
        }
        return $total;
    }

    // retorna los articulos que estan proximos a expirar
    public function articulosVencidos()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(30)->toDateString();
        return  LoteStock::whereBetween('fecha_caducidad', [$fecha1, $fecha2])
            ->get();
    }

    // retorna articulos con minimo de stock
    public function articulosminimo()
    {
        $articulos = Articulo::all();
        $minimos = array();
        foreach ($articulos as $articulo) {
            if ($articulo->stock < $articulo->min_stock) {
                array_push($minimos, $articulo);
            }
        }
        return $minimos;
    }

    // retorna el pago de las facturas
    public function pagosFac()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(15)->toDateString();
        $pagos = PagoFactura::whereBetween('fecha_pago', [$fecha1, $fecha2])
            ->get();
        return $pagos;
    }

    // retorna insumos proximis a expirar
    public function insumosexpiraso()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(30)->toDateString();
        return  LoteInsumo::whereBetween('fecha_vencimiento', [$fecha1, $fecha2])
            ->get();
    }

    // retorna insumos con minimo de stock
    public function insumosminimo()
    {
        $insumos = Insumo::all();
        $minimos = array();
        foreach ($insumos as $articulo) {
            if ($articulo->cantidad < $articulo->minimo) {
                array_push($minimos, $articulo);
            }
        }
        return $minimos;
    }

    // retorna las vacaciones de los usuarios
    public function vacaciones()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(45)->toDateString();
        return  User::whereBetween('vacaciones', [$fecha1, $fecha2])
            ->get();
    }

    // retorna el total en caja
    public function totalCaja()
    {
        $abonos =  Abono::where('created_at', 'like', '%' . date('Y-m-d') . '%')
            ->get();
        $ventasDiaMino =
            DB::table('venta_articulos')
            ->whereDate('created_at', date('Y-m-d'))
            ->where('credito', 0)
            ->where('mayorista', 0)
            ->get();
        $totalVentas = 0;
        foreach ($ventasDiaMino as $ven) {
            $totalVentas = $totalVentas + $ven->total;
        }
        $totalAbonos = 0;
        foreach ($abonos as $abo) {
            $totalAbonos = $totalAbonos + $abo->valor;
        }
        return $totalAbonos + $totalVentas;
    }

    // mantenimiento de mobiliario y EQUIPO
    public function mantenimientoMobiliario()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(35)->toDateString();
        return  MobiliarioEquipo::whereBetween('fecha_mantenimiento', [$fecha1, $fecha2])
            ->get();
    }

    // retorna a los pacientes cumpleañeros
    public function cumpleaneros()
    {

        return DB::table('pacientes')
            ->whereMonth('fecha_nacimiento', date('m'))
            ->whereDay('fecha_nacimiento', date('d'))
            ->get();
    }

    // retorna los recordatorios generales
    public function recordatorioGenerales()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(20)->toDateString();
        return RecordatorioGeneral::whereBetween('fecha', [$fecha1, $fecha2])
            ->get();
    }

    // retorna los recordatorios de los pacientes
    public function recordatorioPaciente()
    {
        $fecha1 = now()->toDateString();
        $fecha2 = now()->addDays(20)->toDateString();
        return DB::table('recordatorio_paciente as rec')
            ->join('pacientes as pac', 'rec.paciente_id', '=', 'pac.id')
            ->select('rec.descripcion', 'rec.dias', 'rec.fecha', 'pac.nombre as paNombre')
            ->whereBetween('rec.fecha', [$fecha1, $fecha2])
            ->get();
    }
}
