<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Mail\PacienteAgendaDoctor;
use App\Models\Agenda;
use App\Models\Doctor;
use App\Models\NotaAgenda;
use App\Models\Paciente;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class AgendaController extends Controller
{

    public function index()
    {
        $doctores = Doctor::all(); // Para el select del modal
        $pacientes = Paciente::all(); // Para el select del modal
        $notas = NotaAgenda::all();
        return view('admin.agenda.index', compact('doctores', 'pacientes', 'notas'));
    }

    // ESTA FUNCIÓN ALIMENTA AL CALENDARIO VÍA AJAX
    public function getEvents(Request $request)
    {
        // 1. Iniciamos la consulta
        $query = Agenda::whereDate('fecha', '>=', $request->start)
            ->whereDate('fecha',   '<=', $request->end)
            ->with(['doctor', 'pacienterelation']);

        // 2. Filtro por Doctor (Si se seleccionó uno)
        if ($request->has('doctor_id') && $request->doctor_id != '') {
            $query->where('doctor_id', $request->doctor_id);
        }

        $data = $query->get();
        $events = [];

        foreach ($data as $item) {
            // --- FECHAS ---
            $startDateTime = $item->fecha . ' ' . $item->hora;
            // Calculamos fin de cita (1 hora)
            $endDateTime = \Carbon\Carbon::parse($startDateTime)->addHour()->format('Y-m-d H:i');

            // --- COLORES BASE ---
            $bgColor = $item->doctor->color ?? '#3788d8';
            $textColor = '#ffffff';
            $borderColor = $bgColor;

            // --- LÓGICA DE ESTADO E ICONOS ---
            // Obtenemos el nombre del paciente (Relacional o Texto manual)
            $nombrePaciente = $item->pacienterelation ? $item->pacienterelation->nombre : $item->paciente;

            $iconoEstado = '⏳ '; // Por defecto (Pendiente)

            if ($item->estado == 'REALIZADO') {
                $iconoEstado = '✅ ';
                $borderColor = '#28a745'; // Borde Verde
            } elseif ($item->estado == 'CANCELADO') {
                $iconoEstado = '❌ ';
                $bgColor = '#6c757d';     // Fondo Gris (apagado)
                $borderColor = '#dc3545'; // Borde Rojo
            }

            // --- ARMADO DEL EVENTO ---
            $events[] = [
                'id'    => $item->id,

                // CORRECCIÓN: Aquí concatenamos el Icono + Nombre + Tratamiento
                'title' => $iconoEstado . $nombrePaciente . ' (' . $item->tratamiento . ')',

                'start' => $startDateTime,
                'end'   => $endDateTime,

                // Estilos
                'backgroundColor' => $bgColor,
                'borderColor'     => $borderColor,
                'textColor'       => $textColor,

                // Datos extra para el Modal (JavaScript)
                'extendedProps' => [
                    'doctor_id'     => $item->doctor_id,
                    'paciente_id'   => $item->paciente_id,
                    'observaciones' => $item->observaciones,
                    'estado'        => $item->estado,
                    'tratamiento'   => $item->tratamiento, // Importante para llenar el modal al editar
                    'paciente_text' => $item->paciente     // Por si es un paciente manual
                ]
            ];
        }

        return response()->json($events);
    }
    public function store(Request $request)
    {
        $request->validate([
            'fecha' => 'required|date',
            'hora' => 'required',
            'doctor_id' => 'required'
        ]);

        // Lógica: Si seleccionó un paciente de la lista, guardamos ID y Nombre
        // Si no, guardamos solo el nombre que escribió a mano (paciente_text)
        $pacienteNombre = "Público General";
        if ($request->paciente_id) {
            $p = Paciente::find($request->paciente_id);
            $pacienteNombre = $p->nombre;
        } elseif ($request->paciente_text) {
            $pacienteNombre = $request->paciente_text;
        }

        Agenda::create([
            'doctor_id' => $request->doctor_id,
            'paciente_id' => $request->paciente_id,
            'paciente' => $pacienteNombre, // Campo string legacy
            'tratamiento' => $request->tratamiento ?? 'Consulta General',
            'fecha' => $request->fecha,
            'hora' => $request->hora,
            'observaciones' => $request->observaciones,
            'estado' => 'PENDIENTE'
        ]);

        return response()->json(['success' => 'Cita Agendada']);
    }

    public function update(Request $request, $id)
    {
        $cita = Agenda::find($id);
        if (!$cita) return response()->json(['error' => 'No encontrado'], 404);

        // CASO 1: Viene de Drag & Drop (Solo movemos fecha/hora rápida)
        if ($request->has('fecha_nueva')) {
            $cita->fecha = Carbon::parse($request->fecha_nueva)->format('Y-m-d');
            $cita->hora  = Carbon::parse($request->fecha_nueva)->format('H:i');
        }
        // CASO 2: Viene del Modal (Edición Completa)
        else {
            // Aquí era donde faltaba asignar los campos nuevos
            $cita->fecha = $request->fecha;
            $cita->hora = $request->hora;  // <--- ¡Esto faltaba!

            $cita->doctor_id = $request->doctor_id;
            $cita->tratamiento = $request->tratamiento;
            $cita->observaciones = $request->observaciones;
            $cita->estado = $request->estado;

            // Lógica para actualizar el paciente si lo cambian
            if ($request->paciente_id) {
                $cita->paciente_id = $request->paciente_id;
                $p = Paciente::find($request->paciente_id);
                $cita->paciente = $p->nombre; // Actualizamos el string de respaldo
            } elseif ($request->paciente_text) {
                $cita->paciente_id = null;
                $cita->paciente = $request->paciente_text;
            }
        }

        $cita->save();
        return response()->json(['success' => 'Cita Actualizada']);
    }

    public function destroy($id)
    {
        Agenda::destroy($id);
        return response()->json(['success' => 'Eliminado']);
    }

    function createNota(Request $request)
    {
        NotaAgenda::create($request->all());
        return back()->with(['info' => 'Nota Guardada', 'color' => 'success']);
    }

    function deleteNota(NotaAgenda $nota)
    {
        $nota->delete();
        return back()->with(['info' => 'Nota Eliminada', 'color' => 'danger']);
    }
}
